/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/
/* @file TIME/TIME.h                                                          */
/* @authors WorldSkills France "Electronique" skill team                      */
/* @version 1.0                                                               */
/* @date 2025-09-06                                                           */
/*                                                                            */
/* @brief This file contains functions to measure and compute date and time.  */
/******************************************************************************/

#ifndef __DEF_H_WSFR48_16_TIME__
#define __DEF_H_WSFR48_16_TIME__

/* *************************** STANDARD INCLUDES **************************** */

#include <stdbool.h>

/* **************************** CUSTOM INCLUDES ***************************** */

/* *************************** TYPES DECLARATION **************************** */

typedef enum
{
    E_CALENDAR_UNKNOWN = 0,
    E_CALENDAR_JULIAN,
    E_CALENDAR_GREGORIAN,
    E_CALENDAR_UTC,
    E_NB_CALENDAR
} tTIME_Calendar;

typedef enum
{
    E_MONTH_JANUARY = 0,
    E_MONTH_FEBRUARY,
    E_MONTH_MARCH,
    E_MONTH_APRIL,
    E_MONTH_MAY,
    E_MONTH_JUNE,
    E_MONTH_JULY,
    E_MONTH_AUGUST,
    E_MONTH_SEPTEMBER,
    E_MONTH_OCTOBER,
    E_MONTH_NOVEMBER,
    E_MONTH_DECEMBER,
    E_NB_MONTHS
} tTIME_Months;

typedef enum
{
    E_DAY_01 = 0,
    E_DAY_02,
    E_DAY_03,
    E_DAY_04,
    E_DAY_05,
    E_DAY_06,
    E_DAY_07,
    E_DAY_08,
    E_DAY_09,
    E_DAY_10,
    E_DAY_11,
    E_DAY_12,
    E_DAY_13,
    E_DAY_14,
    E_DAY_15,
    E_DAY_16,
    E_DAY_17,
    E_DAY_18,
    E_DAY_19,
    E_DAY_20,
    E_DAY_21,
    E_DAY_22,
    E_DAY_23,
    E_DAY_24,
    E_DAY_25,
    E_DAY_26,
    E_DAY_27,
    E_DAY_28,
    E_DAY_29,
    E_DAY_30,
    E_DAY_31
} tTIME_Days;

typedef long long tTIME_Timestamp;

typedef struct
{
    bool validity;
    tTIME_Calendar calendar;
    unsigned int year;
    tTIME_Months month;
    tTIME_Days day;
    unsigned int hour;
    unsigned int minute;
    unsigned int second;
} tTIME_CalendarDate;

/* ******************************* CONSTANTS ******************************** */

#define C_UNDEFINED_TIMESTAMP   (7289654399) /* 31/12/2200, 23:59:59 */
#define C_UNDEFINED_DATE_YEAR   (99)
#define C_UNDEFINED_DATE_MONTH  (99)
#define C_UNDEFINED_DATE_DAY    (99)
#define C_UNDEFINED_DATE_HOUR   (99)
#define C_UNDEFINED_DATE_MINUTE (99)
#define C_UNDEFINED_DATE_SECOND (99)

#define C_NB_MS_IN_S            ((double) 1000.0)
#define C_NB_SEC_IN_DAY         ((double)86400.0)
#define C_NB_SEC_IN_HOUR        ((double) 3600.0)
#define C_NB_SEC_IN_MIN         ((double)   60.0)
#define C_EPOCH_YEAR            (1970)
#define C_NB_DAYS_NOTBISSEXTILE ( 365)
#define C_NB_DAYS_BISSEXTILE    ( 366)

#define C_TIME_CALENDAR_NAME_MAXLENGTH (12)
extern const char C_TIME_CALENDAR_LONGNAME[E_NB_CALENDAR][C_TIME_CALENDAR_NAME_MAXLENGTH];
extern const char C_TIME_CALENDAR_SHRTNAME[E_NB_CALENDAR][C_TIME_CALENDAR_NAME_MAXLENGTH];

/* **************************** GLOBAL VARIABLES **************************** */

extern unsigned int TIME_ELAPSED_SINCE_STARTUP;

/* ************************* FUNCTIONS DECLARATION ************************** */

/******************************************************************************/
/* @function TIME_ComputeDate                                                 */
/*                                                                            */
/* @brief Computes the current date and time.                                 */
/* @param [in] calendar Calendar used by the system                           */
/* @param [in] timestamp Current timestamp                                    */
/* @retval Current date and time in specified calendar                        */
/* @req SYS_REQ-0503-001 : Calcul de l’horodatage                             */
/******************************************************************************/
tTIME_CalendarDate TIME_ComputeDate
(
    const tTIME_Calendar calendar,
    const tTIME_Timestamp timestamp
);

/******************************************************************************/
/* @function TIME_GetTimeElapsedSinceStartup                                  */
/*                                                                            */
/* @brief Gets the number of milliseconds elapsed since system startup.       */
/* @retval Number of milliseconds elapsed since system startup.               */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req SYS_REQ-0501-002 : Temps écoulé depuis le démarrage du système        */
/******************************************************************************/
tTIME_Timestamp TIME_GetTimeElapsedSinceStartup(void);

/******************************************************************************/
/* @function TIME_IncrementTimeElapsedSinceStartup                            */
/*                                                                            */
/* @brief Increments the number of milliseconds elapsed since system startup  */
/*        by 1ms.                                                             */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req SYS_REQ-0501-002 : Temps écoulé depuis le démarrage du système        */
/******************************************************************************/
void TIME_IncrementTimeElapsedSinceStartup(void);

/******************************************************************************/
/* @function TIME_GetTimestamp                                                */
/*                                                                            */
/* @brief Gets the current timestamp in seconds.                              */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req SYS_REQ-0502-001 : Calcul du timestamp actuel                         */
/******************************************************************************/
tTIME_Timestamp TIME_GetTimestamp(void);

/******************************************************************************/
/* @function TIME_SetReferenceTimestamp                                       */
/*                                                                            */
/* @brief Sets the reference timestamp.                                       */
/* @param [in] newTimestamp New reference timestamp, in milliseconds          */
/* @pre The time context must have been initialized by TIME_Init.             */
/* @req None                                                                  */
/******************************************************************************/
void TIME_SetReferenceTimestamp(const tTIME_Timestamp newTimestamp);

/******************************************************************************/
/* @function TIME_Init                                                        */
/*                                                                            */
/* @brief Initializes the time context.                                       */
/* @req None                                                                  */
/******************************************************************************/
void TIME_Init(void);

/******************************************************************************/
/*                                                                            */
/*                             WorldSkills France                             */
/*                       48th edition - Marseille 2025                        */
/*                       Electronique (16) - SyncOrSink                       */
/*                                                                            */
/******************************************************************************/

#endif /* __DEF_H_WSFR48_16_TIME__ */
